<?php
class AdvancedCache {
    private static $instance;
    private $cache = [];
    private $defaultTTL = 3600;
    
    public static function getInstance() {
        if (!self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // تنظيف البيانات المنتهية تلقائياً
        register_shutdown_function([$this, 'clearExpired']);
    }
    
    public function set($key, $data, $ttl = null) {
        if ($ttl === null) {
            $ttl = $this->defaultTTL;
        }
        
        $this->cache[$key] = [
            'data' => $data,
            'expire' => time() + $ttl,
            'created' => time()
        ];
        
        return true;
    }
    
    public function get($key) {
        if (!isset($this->cache[$key])) {
            return null;
        }
        
        $item = $this->cache[$key];
        
        if ($item['expire'] < time()) {
            unset($this->cache[$key]);
            return null;
        }
        
        return $item['data'];
    }
    
    public function delete($key) {
        unset($this->cache[$key]);
        return true;
    }
    
    public function clear() {
        $this->cache = [];
        return true;
    }
    
    public function clearExpired() {
        $currentTime = time();
        $cleared = 0;
        
        foreach ($this->cache as $key => $item) {
            if ($item['expire'] < $currentTime) {
                unset($this->cache[$key]);
                $cleared++;
            }
        }
        
        return $cleared;
    }
    
    public function getStats() {
        $total = count($this->cache);
        $expired = 0;
        $totalSize = 0;
        
        foreach ($this->cache as $item) {
            $itemSize = strlen(serialize($item['data']));
            $totalSize += $itemSize;
            
            if ($item['expire'] < time()) {
                $expired++;
            }
        }
        
        return [
            'total_items' => $total,
            'expired_items' => $expired,
            'active_items' => $total - $expired,
            'memory_usage' => $this->formatBytes($totalSize),
            'hit_rate' => $this->calculateHitRate()
        ];
    }
    
    private function calculateHitRate() {
        // يمكن تنفيذ نظام تتبع الزيارات هنا
        return 0;
    }
    
    private function formatBytes($bytes) {
        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= pow(1024, $pow);
        
        return round($bytes, 2) . ' ' . $units[$pow];
    }
    
    public function getKeys($pattern = null) {
        $keys = array_keys($this->cache);
        
        if ($pattern) {
            $keys = array_filter($keys, function($key) use ($pattern) {
                return fnmatch($pattern, $key);
            });
        }
        
        return $keys;
    }
}
?>