<?php
class AsyncProcessor {
    private static $instance;
    private $queue = [];
    private $processing = false;
    private $batchSize = 50;
    private $maxQueueSize = 1000;
    private $lastProcessTime = 0;
    
    public static function getInstance() {
        if (!self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // تسجيل دالة الإغلاق
        register_shutdown_function([$this, 'shutdown']);
        
        // معالجة الطابور تلقائياً كل 5 ثواني إذا كان هناك مهام
        if (!empty($this->queue)) {
            $this->processBatch();
        }
    }
    
    public function addTask($type, $data, $priority = 'normal') {
        PerformanceMonitor::getInstance()->incrementMetric('async_tasks_added');
        
        // التحقق من حجم الطابور
        if (count($this->queue) >= $this->maxQueueSize) {
            $this->processBatch(); // معالجة فورية إذا كانت الطابور ممتلئة
        }
        
        $task = [
            'type' => $type,
            'data' => $data,
            'priority' => $priority,
            'timestamp' => time(),
            'id' => uniqid()
        ];
        
        // إضافة المهمة حسب الأولوية
        if ($priority === 'high') {
            array_unshift($this->queue, $task);
        } else {
            $this->queue[] = $task;
        }
        
        // معالجة فورية للمهام العالية الأولوية أو عند وصول حجم معين
        if ($priority === 'high' || count($this->queue) >= $this->batchSize) {
            $this->processBatch();
        }
        
        return $task['id'];
    }
    
    private function processBatch() {
        if ($this->processing || empty($this->queue)) {
            return;
        }
        
        $this->processing = true;
        $this->lastProcessTime = time();
        
        try {
            DataManager::startBatch();
            
            // أخذ مجموعة من المهام
            $batch = array_splice($this->queue, 0, $this->batchSize);
            $processed = 0;
            $failed = 0;
            
            foreach ($batch as $task) {
                if ($this->processTask($task)) {
                    $processed++;
                } else {
                    $failed++;
                    // إعادة المهمة الفاشلة إلى نهاية الطابور
                    $this->queue[] = $task;
                }
            }
            
            DataManager::commitBatch();
            
            PerformanceMonitor::getInstance()->incrementMetric('async_batches_processed');
            PerformanceMonitor::getInstance()->incrementMetricBy('async_tasks_processed', $processed);
            
            if ($failed > 0) {
                error_log("AsyncProcessor: $failed tasks failed in batch");
            }
            
        } catch (Exception $e) {
            error_log("Error processing async batch: " . $e->getMessage());
            // إعادة المهام إلى الطابور في حالة الخطأ
            $this->queue = array_merge($batch, $this->queue);
        }
        
        $this->processing = false;
    }
    
    private function processTask($task) {
        try {
            switch ($task['type']) {
                case 'save_points':
                    $points = DataManager::load("storage/data/points.json", false);
                    $points[$task['data']['id']] = $task['data']['points'];
                    return DataManager::save("storage/data/points.json", $points, false);
                    
                case 'update_stats':
                    $stats = DataManager::load("storage/data/stats.json", false);
                    
                    // تحديث الإحصائيات
                    if (!isset($stats['all'])) {
                        $stats['all'] = ['trybuy' => 0, 'buy' => 0];
                    }
                    if (!isset($stats[$task['data']['id']])) {
                        $stats[$task['data']['id']] = ['buy' => 0];
                    }
                    
                    $stats['all']['trybuy'] += $task['data']['trybuy'] ?? 0;
                    $stats['all']['buy'] += $task['data']['buy'] ?? 0;
                    $stats[$task['data']['id']]['buy'] += $task['data']['user_buy'] ?? 0;
                    
                    return DataManager::save("storage/data/stats.json", $stats, false);
                    
                case 'log_operation':
                    $op = DataManager::load("storage/data/operations.json", false);
                    $op[] = $task['data'];
                    return DataManager::save("storage/data/operations.json", $op, false);
                    
                case 'update_invite':
                    $invite = DataManager::load("storage/data/invites.json", false);
                    $invite['whoInvitedMe'][$task['data']['invited_id']] = $task['data']['inviter_id'];
                    $invite['invited'][$task['data']['inviter_id']][] = $task['data']['invited_id'];
                    return DataManager::save("storage/data/invites.json", $invite, false);
                    
                default:
                    error_log("Unknown async task type: " . $task['type']);
                    return false;
            }
        } catch (Exception $e) {
            error_log("Error processing async task {$task['type']}: " . $e->getMessage());
            return false;
        }
    }
    
    public function getQueueStats() {
        return [
            'queue_size' => count($this->queue),
            'processing' => $this->processing,
            'last_process_time' => $this->lastProcessTime,
            'batch_size' => $this->batchSize
        ];
    }
    
    public function shutdown() {
        if (!empty($this->queue) && !$this->processing) {
            $this->processBatch();
        }
    }
    
    public function clearQueue() {
        $this->queue = [];
        return true;
    }
    
    public function getTaskCountByType() {
        $counts = [];
        foreach ($this->queue as $task) {
            $type = $task['type'];
            $counts[$type] = ($counts[$type] ?? 0) + 1;
        }
        return $counts;
    }
}
?>