<?php
class DataManager {
    private static $cache;
    private static $batchMode = false;
    private static $batchData = [];
    private static $locks = [];
    
    public static function init() {
        self::$cache = AdvancedCache::getInstance();
        
        // إنشاء المجلدات اللازمة
        $dirs = [
            'storage/data',
            'storage/cache',
            'storage/logs',
            'storage/backups'
        ];
        
        foreach ($dirs as $dir) {
            if (!is_dir($dir)) {
                mkdir($dir, 0755, true);
            }
        }
        
        // تنظيف الأقفال القديمة عند البدء
        self::cleanOldLocks();
    }
    
    public static function load($file, $useCache = true) {
        $cacheKey = "file_" . md5($file);
        
        // استخدام التخزين المؤقت
        if ($useCache && $cached = self::$cache->get($cacheKey)) {
            return $cached;
        }
        
        // التحقق من وجود الملف
        if (!file_exists($file)) {
            $data = [];
            // إنشاء ملف جديد إذا لم يكن موجوداً
            file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT));
            self::$cache->set($cacheKey, $data, 300);
            return $data;
        }
        
        // محاولة قراءة الملف
        $attempts = 3;
        while ($attempts > 0) {
            try {
                $content = file_get_contents($file);
                if ($content === false) {
                    throw new Exception("Failed to read file: $file");
                }
                
                $data = json_decode($content, true) ?? [];
                break;
            } catch (Exception $e) {
                $attempts--;
                if ($attempts === 0) {
                    error_log("Failed to load file after 3 attempts: $file - " . $e->getMessage());
                    $data = [];
                } else {
                    usleep(100000); // انتظار 100ms
                }
            }
        }
        
        if ($useCache) {
            self::$cache->set($cacheKey, $data, 300); // 5 دقائق
        }
        
        return $data;
    }
    
    public static function save($file, $data, $immediate = true) {
        if (self::$batchMode && !$immediate) {
            self::$batchData[$file] = $data;
            return true;
        }
        
        $lockFile = $file . '.lock';
        $maxWait = 5; // أقصى انتظار 5 ثواني
        $startTime = time();
        
        // الحصول على قفل الملف
        while (file_exists($lockFile) && (time() - $startTime) < $maxWait) {
            // التحقق مما إذا كان القفل قديمًا (أكثر من 30 ثانية)
            if (filemtime($lockFile) < (time() - 30)) {
                unlink($lockFile);
                break;
            }
            usleep(100000); // انتظار 100ms
        }
        
        if (file_exists($lockFile)) {
            error_log("Failed to acquire lock for file: $file");
            return false;
        }
        
        // إنشاء القفل
        file_put_contents($lockFile, getmypid() . ':' . time());
        
        try {
            // استخدام ملف مؤقت للكتابة
            $tempFile = $file . '.tmp.' . uniqid();
            $json = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            
            if (file_put_contents($tempFile, $json) === false) {
                throw new Exception("Failed to write temporary file");
            }
            
            // التأكد من كتابة البيانات
            if (file_get_contents($tempFile) !== $json) {
                throw new Exception("Data verification failed");
            }
            
            // نقل الملف المؤقت إلى الملف الرئيسي
            if (!rename($tempFile, $file)) {
                throw new Exception("Failed to move temporary file");
            }
            
            // تحديث التخزين المؤقت
            $cacheKey = "file_" . md5($file);
            self::$cache->set($cacheKey, $data, 300);
            
            // إزالة القفل
            unlink($lockFile);
            
            return true;
            
        } catch (Exception $e) {
            // تنظيف في حالة الخطأ
            @unlink($tempFile);
            @unlink($lockFile);
            error_log("Error saving file $file: " . $e->getMessage());
            return false;
        }
    }
    
    public static function startBatch() {
        self::$batchMode = true;
        self::$batchData = [];
    }
    
    public static function commitBatch() {
        $success = true;
        
        foreach (self::$batchData as $file => $data) {
            if (!self::save($file, $data, true)) {
                $success = false;
                error_log("Failed to save file in batch: $file");
            }
        }
        
        self::$batchMode = false;
        self::$batchData = [];
        
        return $success;
    }
    
    public static function backup($file) {
        if (!file_exists($file)) {
            return false;
        }
        
        $backupDir = 'storage/backups/' . date('Y-m-d');
        if (!is_dir($backupDir)) {
            mkdir($backupDir, 0755, true);
        }
        
        $backupFile = $backupDir . '/' . basename($file) . '.' . date('H-i-s') . '.bak';
        return copy($file, $backupFile);
    }
    
    private static function cleanOldLocks() {
        $files = glob('storage/data/*.lock');
        $currentTime = time();
        
        foreach ($files as $lockFile) {
            if (filemtime($lockFile) < ($currentTime - 300)) { // أقفال أقدم من 5 دقائق
                unlink($lockFile);
            }
        }
    }
    
    public static function getMemoryUsage() {
        $usage = memory_get_usage(true);
        $peak = memory_get_peak_usage(true);
        
        return [
            'current' => self::formatBytes($usage),
            'peak' => self::formatBytes($peak)
        ];
    }
    
    private static function formatBytes($bytes) {
        $units = ['B', 'KB', 'MB', 'GB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        $bytes /= pow(1024, $pow);
        
        return round($bytes, 2) . ' ' . $units[$pow];
    }
    
    public static function getStorageStats() {
        $dataFiles = glob('storage/data/*.json');
        $totalSize = 0;
        $fileCount = 0;
        
        foreach ($dataFiles as $file) {
            if (file_exists($file)) {
                $totalSize += filesize($file);
                $fileCount++;
            }
        }
        
        return [
            'file_count' => $fileCount,
            'total_size' => self::formatBytes($totalSize),
            'average_size' => $fileCount > 0 ? self::formatBytes($totalSize / $fileCount) : '0 B'
        ];
    }
}
?>