<?php
class HighPerformanceApi {
    private $api_key;
    private $url = "https://api.spider-service.com?apiKay=";
    private $cache;
    private $retryAttempts = 3;
    private $retryDelay = 1;
    private $timeout = 30;
    
    public function __construct($api_key) {
        $this->api_key = $api_key;
        $this->url .= $api_key;
        $this->cache = AdvancedCache::getInstance();
        
        // التحقق من صحة API key
        if (empty($api_key)) {
            throw new InvalidArgumentException("API key cannot be empty");
        }
    }
    
    private function getData($data) {
        $url = $this->url;
        foreach ($data as $k => $v) {
            $url .= "&{$k}=" . urlencode($v);
        }
        
        $cacheKey = "api_" . md5($url);
        
        // التحقق من التخزين المؤقت للطلبات غير المتغيرة
        if (isset($data['action']) && in_array($data['action'], ['getBalance', 'getCountrys'])) {
            if ($cached = $this->cache->get($cacheKey)) {
                PerformanceMonitor::getInstance()->incrementMetric('api_cache_hits');
                return $cached;
            }
        }
        
        PerformanceMonitor::getInstance()->incrementMetric('api_requests');
        
        $context = stream_context_create([
            'http' => [
                'timeout' => $this->timeout,
                'header' => "User-Agent: TelegramBot/1.0\r\n"
            ]
        ]);
        
        $response = $this->retryRequest(function() use ($url, $context) {
            $result = file_get_contents($url, false, $context);
            if ($result === false) {
                throw new Exception("Failed to fetch data from API");
            }
            return $result;
        });
        
        if ($response === "error") {
            return "error";
        }
        
        $decoded = json_decode($response, true);
        
        // تخزين النتائج المؤقتة
        if (isset($data['action']) && in_array($data['action'], ['getBalance', 'getCountrys'])) {
            $ttl = $data['action'] === 'getBalance' ? 300 : 1800; // 5 دقائق للرصيد، 30 دقيقة للدول
            $this->cache->set($cacheKey, $decoded, $ttl);
        }
        
        return $decoded;
    }
    
    public function getBalance() {
        $getData = $this->getData(['action' => 'getBalance']);
        
        if (isset($getData['error']) && $getData['error'] == 'INFORMATION_SUCCESS') {
            return $getData['result']['wallet'] ?? 0;
        } else {
            error_log("API Balance Error: " . ($getData['error'] ?? 'Unknown error'));
            return 0;
        }
    }
    
    public function getNumber($countryCode) {
        // التحقق من توفر الدولة مسبقاً
        $availabilityKey = "country_available_{$countryCode}";
        if ($this->cache->get($availabilityKey) === false) {
            return "error";
        }
        
        $getData = $this->getData([
            'action' => 'getNumber',
            'country' => $countryCode
        ]);
        
        if (isset($getData['error']) && $getData['error'] == 'INFORMATION_SUCCESS') {
            return [
                'number' => $getData['result']['phone'],
                'hash_code' => $getData['result']['hash_code'],
            ];
        } else {
            // تحديث حالة التوفر
            $this->cache->set($availabilityKey, false, 60); // دقيقة واحدة
            error_log("API GetNumber Error for {$countryCode}: " . ($getData['error'] ?? 'Unknown error'));
            return "error";
        }
    }
    
    public function getCode($hashCode) {
        $getData = $this->getData([
            'action' => 'getCode',
            'hash_code' => $hashCode
        ]);
        
        if (isset($getData['error']) && $getData['error'] == 'INFORMATION_SUCCESS') {
            return $getData['result'];
        } else {
            error_log("API GetCode Error for hash {$hashCode}: " . ($getData['error'] ?? 'Unknown error'));
            return "error";
        }
    }
    
    public function getCountries() {
        $getData = $this->getData(['action' => 'getCountrys']);
        
        if (isset($getData['error']) && $getData['error'] == 'INFORMATION_SUCCESS') {
            return $getData['result']['countries'][1] ?? [];
        } else {
            error_log("API GetCountries Error: " . ($getData['error'] ?? 'Unknown error'));
            return "error";
        }
    }
    
    public function preloadData() {
        // تحميل مسبق للبيانات الشائعة الاستخدام
        $this->getCountries();
        $this->getBalance();
    }
    
    public function getApiStatus() {
        $testData = $this->getData(['action' => 'getBalance']);
        return isset($testData['error']) && $testData['error'] == 'INFORMATION_SUCCESS';
    }
    
    private function retryRequest($callback) {
        $attempts = 0;
        
        while ($attempts < $this->retryAttempts) {
            try {
                $result = $callback();
                return $result;
            } catch (Exception $e) {
                $attempts++;
                error_log("API Request failed (attempt $attempts): " . $e->getMessage());
                
                if ($attempts < $this->retryAttempts) {
                    sleep($this->retryDelay * $attempts);
                }
            }
        }
        
        return "error";
    }
    
    public function getPerformanceStats() {
        return [
            'cache_hits' => PerformanceMonitor::getInstance()->getMetric('api_cache_hits'),
            'total_requests' => PerformanceMonitor::getInstance()->getMetric('api_requests'),
            'api_status' => $this->getApiStatus()
        ];
    }
}
?>