<?php
class PerformanceMonitor {
    private static $instance;
    private $startTime;
    private $memoryStart;
    private $metrics = [];
    private $requestId;
    
    public static function getInstance() {
        if (!self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->requestId = uniqid();
    }
    
    public function startRequest() {
        $this->startTime = microtime(true);
        $this->memoryStart = memory_get_usage(true);
        $this->metrics = [
            'bot_requests' => 0,
            'api_requests' => 0,
            'api_cache_hits' => 0,
            'async_tasks_added' => 0,
            'async_tasks_processed' => 0,
            'async_batches_processed' => 0,
            'file_reads' => 0,
            'file_writes' => 0,
            'cache_hits' => 0,
            'cache_misses' => 0
        ];
        
        return $this->requestId;
    }
    
    public function endRequest() {
        $endTime = microtime(true);
        $endMemory = memory_get_usage(true);
        
        $performanceData = [
            'request_id' => $this->requestId,
            'execution_time' => round(($endTime - $this->startTime) * 1000, 2),
            'memory_used' => $endMemory - $this->memoryStart,
            'peak_memory' => memory_get_peak_usage(true),
            'metrics' => $this->metrics,
            'timestamp' => time(),
            'url' => $_SERVER['REQUEST_URI'] ?? 'N/A',
            'method' => $_SERVER['REQUEST_METHOD'] ?? 'N/A'
        ];
        
        // تسجيل البيانات إذا تجاوز وقت التنفيذ حداً معيناً
        if ($performanceData['execution_time'] > 1000) { // أكثر من ثانية
            $this->logPerformance($performanceData);
        }
        
        return $performanceData;
    }
    
    public function incrementMetric($metric, $value = 1) {
        if (isset($this->metrics[$metric])) {
            $this->metrics[$metric] += $value;
        } else {
            $this->metrics[$metric] = $value;
        }
    }
    
    public function incrementMetricBy($metric, $value) {
        $this->incrementMetric($metric, $value);
    }
    
    public function getMetric($metric) {
        return $this->metrics[$metric] ?? 0;
    }
    
    private function logPerformance($data) {
        $logDir = 'storage/logs';
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        $logFile = $logDir . '/performance.log';
        $logEntry = [
            'time' => date('Y-m-d H:i:s'),
            'data' => $data
        ];
        
        file_put_contents($logFile, json_encode($logEntry) . PHP_EOL, FILE_APPEND | LOCK_EX);
    }
    
    public function getPerformanceReport($hours = 24) {
        $logFile = 'storage/logs/performance.log';
        if (!file_exists($logFile)) {
            return ['error' => 'No performance data available'];
        }
        
        $cutoffTime = time() - ($hours * 3600);
        $lines = file($logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $recentLogs = [];
        
        foreach ($lines as $line) {
            $data = json_decode($line, true);
            if ($data && isset($data['data']['timestamp']) && $data['data']['timestamp'] >= $cutoffTime) {
                $recentLogs[] = $data['data'];
            }
        }
        
        if (empty($recentLogs)) {
            return ['message' => 'No recent performance data'];
        }
        
        $report = [
            'total_requests' => count($recentLogs),
            'avg_execution_time' => 0,
            'max_execution_time' => 0,
            'avg_memory_used' => 0,
            'max_memory_used' => 0,
            'total_bot_requests' => 0,
            'total_api_requests' => 0,
            'cache_efficiency' => 0,
            'slow_requests' => []
        ];
        
        $totalTime = 0;
        $totalMemory = 0;
        $totalCacheHits = 0;
        $totalCacheCalls = 0;
        
        foreach ($recentLogs as $log) {
            $totalTime += $log['execution_time'];
            $totalMemory += $log['memory_used'];
            $totalCacheHits += $log['metrics']['api_cache_hits'] ?? 0;
            $totalCacheCalls += ($log['metrics']['api_cache_hits'] ?? 0) + ($log['metrics']['api_requests'] ?? 0);
            
            $report['total_bot_requests'] += $log['metrics']['bot_requests'] ?? 0;
            $report['total_api_requests'] += $log['metrics']['api_requests'] ?? 0;
            
            if ($log['execution_time'] > $report['max_execution_time']) {
                $report['max_execution_time'] = $log['execution_time'];
            }
            
            if ($log['memory_used'] > $report['max_memory_used']) {
                $report['max_memory_used'] = $log['memory_used'];
            }
            
            // تحديد الطلبات البطيئة
            if ($log['execution_time'] > 2000) { // أكثر من ثانيتين
                $report['slow_requests'][] = [
                    'request_id' => $log['request_id'],
                    'execution_time' => $log['execution_time'],
                    'url' => $log['url']
                ];
            }
        }
        
        if (count($recentLogs) > 0) {
            $report['avg_execution_time'] = round($totalTime / count($recentLogs), 2);
            $report['avg_memory_used'] = round($totalMemory / count($recentLogs));
            $report['cache_efficiency'] = $totalCacheCalls > 0 ? round(($totalCacheHits / $totalCacheCalls) * 100, 2) : 0;
        }
        
        return $report;
    }
    
    public function getSystemStats() {
        $cache = AdvancedCache::getInstance();
        $dataManagerStats = DataManager::getStorageStats();
        $cacheStats = $cache->getStats();
        
        return [
            'memory_usage' => DataManager::getMemoryUsage(),
            'storage_stats' => $dataManagerStats,
            'cache_stats' => $cacheStats,
            'system_load' => sys_getloadavg(),
            'php_version' => PHP_VERSION,
            'max_execution_time' => ini_get('max_execution_time'),
            'memory_limit' => ini_get('memory_limit')
        ];
    }
    
    public function cleanupOldLogs($days = 7) {
        $logDir = 'storage/logs';
        $cutoffTime = time() - ($days * 86400);
        
        $files = glob($logDir . '/*.log');
        $deleted = 0;
        
        foreach ($files as $file) {
            if (filemtime($file) < $cutoffTime) {
                unlink($file);
                $deleted++;
            }
        }
        
        return $deleted;
    }
}
?>