<?php
class TranslationManager {
    private static $instance;
    private $cache;
    private $loadedLanguages = [];
    private $defaultLang = 'ar';
    
    public static function getInstance() {
        if (!self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->cache = AdvancedCache::getInstance();
    }
    
    public function loadLanguage($lang) {
        // استخدام اللغة الافتراضية إذا لم يتم توفير لغة
        if (empty($lang)) {
            $lang = $this->defaultLang;
        }
        
        // التحقق من التخزين المؤقت في الذاكرة
        if (isset($this->loadedLanguages[$lang])) {
            return $this->loadedLanguages[$lang];
        }
        
        // التحقق من التخزين المؤقت العام
        $cacheKey = "lang_{$lang}";
        if ($cached = $this->cache->get($cacheKey)) {
            $this->loadedLanguages[$lang] = $cached;
            return $cached;
        }
        
        // تحميل ملف اللغة
        $langFile = "langs/{$lang}.php";
        if (file_exists($langFile)) {
            $translations = include $langFile;
            
            // التحقق من صحة البيانات
            if (is_array($translations)) {
                $this->loadedLanguages[$lang] = $translations;
                $this->cache->set($cacheKey, $translations, 3600); // ساعة واحدة
                return $translations;
            }
        }
        
        // استخدام اللغة الافتراضية إذا فشل التحميل
        if ($lang !== $this->defaultLang) {
            error_log("Failed to load language: $lang, falling back to default");
            return $this->loadLanguage($this->defaultLang);
        }
        
        // اللغة الافتراضية غير موجودة
        error_log("Default language file not found: {$this->defaultLang}.php");
        return [];
    }
    
    public function get($key, $lang, $replacements = []) {
        $translations = $this->loadLanguage($lang);
        
        // البحث عن النص المترجم
        $text = $translations[$key] ?? $key;
        
        // استبدال العناصر النائبة
        foreach ($replacements as $placeholder => $value) {
            $text = str_replace($placeholder, $value, $text);
        }
        
        return $text;
    }
    
    public function getCountryName($countryCode, $lang) {
        $cacheKey = "country_names_{$lang}";
        
        if (!$countryNames = $this->cache->get($cacheKey)) {
            $countryFile = "config/country_names.php";
            if (file_exists($countryFile)) {
                $allNames = include $countryFile;
                $countryNames = $allNames[$lang] ?? $allNames['ar'] ?? [];
                $this->cache->set($cacheKey, $countryNames, 86400); // 24 ساعة
            } else {
                $countryNames = [];
            }
        }
        
        return $countryNames[$countryCode] ?? $countryCode;
    }
    
    public function setDefaultLanguage($lang) {
        $this->defaultLang = $lang;
    }
    
    public function getLoadedLanguages() {
        return array_keys($this->loadedLanguages);
    }
    
    public function clearCache($lang = null) {
        if ($lang) {
            $this->cache->delete("lang_{$lang}");
            unset($this->loadedLanguages[$lang]);
        } else {
            foreach ($this->loadedLanguages as $langCode => $data) {
                $this->cache->delete("lang_{$langCode}");
            }
            $this->loadedLanguages = [];
        }
        
        return true;
    }
    
    public function getLanguageStats() {
        $langFiles = glob('langs/*.php');
        $stats = [];
        
        foreach ($langFiles as $file) {
            $lang = basename($file, '.php');
            $translations = $this->loadLanguage($lang);
            $stats[$lang] = [
                'translation_count' => count($translations),
                'loaded' => isset($this->loadedLanguages[$lang]),
                'file_size' => filesize($file)
            ];
        }
        
        return $stats;
    }
}
?>